/*
Scripts File
Purposes:  
    
	Functions that perform Daylight Simulation Analysis-related actions.	

Revision History

	November 5, 2007: David Cunningham
	
*/

fn CreateDefaultDaylightSimulationMaterial =
(
	-- either replace this return value here, or define your 
	-- own CreateDefaultDaylightSimulationMaterial function in the
	-- Listener window
	Arch___Design__mi()  
)


fn FixInvalidMaterialsForDaylightSimulation =
(
	-- fix selected nodes with a standard Architectural and Design Material
	for i in $selection do
	(
		-- Change this material if you are not happy with the default assignment
		i.material = CreateDefaultDaylightSimulationMaterial()
		i.material.refl_weight = 0.0
		i.material.name = ~MATERIAL_NAME_LIGHTING_ANALYSIS_DEFAULT_MATERIAL~
	)
)

fn CreateValidDaylightSimulationLightCallback n = 
(
	if classof n  != TargetObject do
	(
		n.baseObject.castShadows = true
		n.shadowGenerator = raytraceShadow()
	)
)

struct DontShowAgainDialog
(
	titleStr = undefined,
	messageStr = undefined,
	retval = undefined,
	notAgain = undefined,
	default = undefined,
	
	fn DoDialog =
	(
		if GetQuietMode() then
		(
			if retval==undefined do retval = default
		)
		else if notAgain!=true do
		(
			-- HACK:  MaxScript has no general mechanism to "Don't Show This Again" dialogs.
			-- And it has no means to change the title and labels within a dialog before displaying it.
			-- So, dialog handling here is manual.
			-- To support arbitrary titles and labels, the dialog is defined as a string, and the values are printed into it.
			local rolloutTemplate =
				"rollout DontShowAgainRollout \"%\" width:564 height:202
				(
					--LOC_NOTES:localize the three message below, but don't change the unusual quote marks
					button bNo ~BNO_CAPTION~ pos:[286,144] width:90 height:24
					button bYes ~BYES_CAPTION~ pos:[188,144] width:90 height:24
					checkbox cNotAgain ~CNOTAGAIN_CAPTION~ pos:[8,176] width:360 height:15 checked:false
					label eLabel \"%\" pos:[8,10] width:532 height:130
					local retval = undefined
					local notAgain = undefined

					on bNo pressed do
					( 
						retval = false
						notAgain = cNotAgain.checked
						destroydialog DontShowAgainRollout
					)
					on bYes pressed do
					(
						retval = true
						notAgain = cNotAgain.checked
						destroydialog DontShowAgainRollout
					)
				)"
			
			-- Print title and labels into the rollout definition
			local rolloutStringStream = StringStream ""
			format rolloutTemplate (titleStr as string) (messageStr as string) to:rolloutStringStream
			
			-- Execute the rollout definition
			seek rolloutStringStream 0
			local rolloutValue = execute rolloutStringStream
			
			-- Display the dialog
			CreateDialog rolloutValue modal:true
			
			-- Get the return values
			notAgain = rolloutValue.notAgain
			retval = rolloutValue.retval
		)
		retval
	)
)

-- Checks if an appropriate exposure control is currently applied, for this light type.
-- If not, the user is prompted with the option to automatically apply an approriate exposure control.
global ValidateExposureControlForLight_Dialog = DontShowAgainDialog()

fn ValidateExposureControlForLight lightClass = 
(

	local isDaylight = (lightClass == Daylight)
	local isPhotometric = ((lightClass == Target_Light) or (lightClass == Free_Light))

	local isApplicableLight = (isDaylight or isPhotometric)

	if ((not isApplicableLight)) do
	( -- Not creating a relvant light type.  No changes needed.
		return()
	)
	
	local currentExpCtrl = sceneExposureControl.exposureControl
	local isNullExpCtrl = (currentExpCtrl==undefined)
	compatibleExpCtrl = Physical_Camera_Exposure_Control

	local isCompatibleExpCtrl = ((not isNullExpCtrl) and ((classof currentExpCtrl) == compatibleExpCtrl))
	local isOtherExpCtrl = ((not isNullExpCtrl) and (not isCompatibleExpCtrl))

	-- If the user agreed, update the exposure control.
	local createExposure = GetQuietMode()
	
	--always create the exposure control if we are in quiet mode.
	if (not createExposure) do
	(
		local titleStr
		local messageStr

		-- Validate the current exposure control.
		if isDaylight then
		(
			-- Don't prompt the user for changes if (1) the standard exposure control is applied and perfectly configured, or (2) the user applied their own type of exposure control.

			local isDesiredExpCtrl = (isCompatibleExpCtrl and (currentExpCtrl.global_ev == 15))
			if (isDesiredExpCtrl or isOtherExpCtrl) do return() -- No changes needed.
			--LOC_NOTES:localize this. Use && for an ampersand character
			messageStr = ~MSGSTR_RECOMMENDED_PHYSICAL_EXPOSURE_CONTROL~ +
				~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_EE_DIALOG~		

			--LOC_NOTES:localize this
			titleStr = ~TITLESTR_DAYLIGHT_SYSTEM_CREATION~
		)
		else if isPhotometric then
		(
			if not isNullExpCtrl do return () -- Not creating a Daylight, and exposure control is already applied.  No changes needed.

			--LOC_NOTES:localize this. Use && for an ampersand character
			messageStr = ~MSGSTR_PHOTOMETRIC_LIGHT_RECOMMENDED_PHYSICAL_CAMERA_EXPOSURE_CONTROL~ +
				~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_ENV_AND_EFFECTS_DIALOG~

			--LOC_NOTES:localize this
			titleStr = ~TITLESTR_PHOTOMETRIC_LIGHT_CREATION~
		)
		else return() -- Not any applicable light type

		-- Checks failed.  Prompt user.
		ValidateExposureControlForLight_Dialog.titleStr = titleStr
		ValidateExposureControlForLight_Dialog.messageStr = messageStr
		ValidateExposureControlForLight_Dialog.default = false
		ValidateExposureControlForLight_Dialog.DoDialog()
		createExposure = ValidateExposureControlForLight_Dialog.retval
	)
	
	if createExposure do
	(
		-- Create a new exposure control if there is no current one
		if (isNullExpCtrl) do 
		(
			sceneExposureControl.exposureControl = compatibleExpCtrl()
			currentExpCtrl = sceneExposureControl.exposureControl
		)
		
		if isDaylight then
		(
			currentExpCtrl.global_ev = 15
		)
		else if isPhotometric do
		(
			currentExpCtrl.global_ev = 6
		)
	)
)


-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQAr4WJc2mEs5CafJPJ4J20QlnazAA5xGOrp
-- 1lWR04j/9BScipiiGq15V9ExU5A200eh6DRVPYdFoQKboMI/s37Y/2dknfU0eE5q
-- ROxQqUDC1WBfY+U4xdLV4wAgAtVgWNVKZsJDxUFIC+v4rp3ermBvyZDx1SJ00ReV
-- HZh3f8GhOD/2pAr15l/ylEHEEr6nW2ZnCTPQJ96fCQzcZ47pnidJTAIjms1G8rLO
-- q1FRetu4RtHQiBrxe+5TmG7e+iYhhh736dN935q0rrHqvbM2ry63R84ljRwUlFOj
-- QYlaOdXt1EampwgFci5s1fhIlt4FAFxzSnf/zsS55f5DLqZicomH
-- -----END-SIGNATURE-----